// 
//  Copyright (C) 1995,2007,2010,2017, 2019  Smithsonian Astrophysical Observatory
//
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with this program; if not, write to the Free Software Foundation, Inc.,
//  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//

/*H****************************************************************************
 
* FILE NAME: %M%
 
* DEVELOPEMENT: UI
 
* DESCRIPTION:
 
        This file contains the class definition for the Host Level 
	Parameter interface. This class incapsulates the IRAF Compatible
	Parameter interface library into a class.
 
* NOTES:
 
        This class is based on the IRAF Compatible Parameter interface.
	This class does not support the structure option.
 
*H****************************************************************************/

#include "ParamItem.hh"

#include <iomanip>
#include <iostream>
#include <sstream>
#include <float.h>
#include <stdlib.h>
#include <limits.h>

using namespace std;

// The max length a parameter entry can be in a file.
const int MAX_PARAM_LENGTH = 512;
const char FORMAT_DELIM = '`';
const char DEFAULT_DELIM = ',';

const double INDEF_REAL = INDEFD;
const int INDEF_INT = INDEFI;

// parameter mode masks 
const int NO_MODE = 0x00;
const int AUTO_MODE = 0x01;
const int HIDDEN_MODE = 0x02;
const int LEARN_MODE = 0x04;
const int QUERY_MODE = 0x08;

// Global method prototypes
void tokenize(string orig, list < string > &stringlist, const char delim);
bool is_signed_numeric(string val);


// ===========================================================================
// Global Methods
// ===========================================================================

// tokenize a string into a string list using the specified delimeter
void tokenize(string orig, list < string > &stringlist, const char delim = ' ')
{
  string::size_type end;
  stringlist.clear ();
  while (orig.length ())
    {
      if ((end = orig.find (delim)) == string::npos)
	end = orig.length () + 1;
      stringlist.push_back (orig.substr (0, end));
      orig.replace (0, end + 1, "");
    }
}

// return true if value is a signed numeric- must contain at least 1 digit
// first char may be optional sign 
bool is_signed_numeric(string val)
{
  unsigned int rr, ll = val.length ();
  bool rv = false;
  for (rr = ll; rr--;)
    {
      if (!isdigit (val[rr]))
	{
	  if (rr || ((val[rr] != '-') && (val[rr] != '+')))
	    return false;
	  else
	    rv = true;
	}
    }
  return rv;
}

int _cxcparam_string_to_double(string val, double &outv)
{
  string tmp = val;
  tmp.erase (tmp.find_last_not_of (" \t") + 1);
  istringstream ss (tmp);
  ss.precision (DBL_DIG);
  return (!((ss >> outv) && ss.eof ()));
}

int _cxcparam_string_to_long(string val, long &outv)
{
  string tmp = val;
  tmp.erase (tmp.find_last_not_of (" \t") + 1);
  istringstream ss (tmp);
  return (!((ss >> outv) && ss.eof ()));
}

// compare 2 strings ignoring case and preceeding/trailing whitespace
int _cxcparam_strcmp_loose (string str1, string str2)
{
  string t1, t2;
  unsigned int ii;
  int diff = 1, last = str1.find_first_not_of (" \t", 0);
  int pos = str1.find_first_of (" \t", last);
  if (last > -1)
    t1 = str1.substr (last, pos - last);
  last = str2.find_first_not_of (" \t", 0);
  pos = str2.find_first_of (" \t", last);
  if (last > -1)
    t2 = str2.substr (last, pos - last);

  if ((ii = t1.length ()) == t2.length ())
    {
      diff = 0;
      while (ii-- && !diff)
	if (tolower (t1[ii]) != tolower (t2[ii]))
	  diff = 1;
    }
  return diff;
}

// ===========================================================================
// Class: ParamItem
// 
// Description:
//   Base class for Parameter records (individual parameter).
// 
// ===========================================================================
// Constructor
ParamItem::ParamItem ()
{
  name.clear ();
  prompt.clear ();
  changed = false;
  delim = DEFAULT_DELIM;
  position = 0;
}

ParamItem::ParamItem (const ParamItem & item)
{
  name = item.name;
  type = item.type;
  mode = item.mode;
  prompt = item.prompt;
  changed = false;
  position = item.position;
  delim = item.delim;
  parfile = item.parfile;
  file_name = item.file_name;
}

void ParamItem::SetFilename (char *filename)
{
  if (filename)
    file_name = filename;
}

// Destructor
ParamItem::~ParamItem ()
{
}

// Returns a char** of the fields in the parameter
char** ParamItem::GetChars (int *count)
{
  char** str = NULL;
  ostringstream oss;

  // Fill oss with formatted parameter content
  SetDelim(FORMAT_DELIM);
  Print(oss);
  SetDelim(DEFAULT_DELIM);  // Reset delimeter

  // Split fields to list
  string tempstr = oss.str();
  list<string> tmp;
  tokenize(tempstr, tmp, FORMAT_DELIM);

  str = new char*[tmp.size() + 1];
  int cnt = 0;
  for (list<string>::iterator ii = tmp.begin(); ii != tmp.end(); ii++)
  {
    str[cnt] = new char[(*ii).length() + 1];
    strcpy( str[cnt++], (*ii).c_str() );
  }
  str[cnt] = 0;

  if (count)
    *count = cnt;

  return(str);
}

// Returns the character lengths of each field in the parameter
short* ParamItem::GetWidths()
{
  ostringstream oss;

  // Fill oss with formatted parameter content
  SetDelim(FORMAT_DELIM);
  Print(oss);
  SetDelim(DEFAULT_DELIM);  // Reset delimeter

  // Split fields to list
  string tempstr = oss.str();
  list<string> tmp;
  tokenize (tempstr, tmp, FORMAT_DELIM);

  int count = 0;
  short* widths = new short[tmp.size()];
  for (list<string>::iterator ii = tmp.begin(); ii != tmp.end(); ii++)
  {
    widths[count++] = (*ii).length();
  }

  return(widths);
}

// Determine the parameter mode mask 
int ParamItem::CalculateMode (char *val)
{
  int md = NO_MODE;
  if (val)
  {
    // first check for words query/learn/auto/hidden
    if (strstr (val, "query") != 0)
      md |= QUERY_MODE;
    if (strstr (val, "learn") != 0)
      md |= LEARN_MODE;
    if (strstr (val, "hidden") != 0)
      md |= HIDDEN_MODE;
    if (strstr (val, "auto") != 0)
      md |= AUTO_MODE;

    if (md == NO_MODE)	// if not set, check for q/l/h/a
      while (*val)
      {
	switch (*val++)
	{
	case 'q':
	  md |= QUERY_MODE;
	  break;
	case 'l':
	  md |= LEARN_MODE;
	  break;
	case 'h':
	  md |= HIDDEN_MODE;
	  break;
	case 'a':
	  md |= AUTO_MODE;
	  break;
	default:
	  break;
	}
      }
  }

  return (md);
}

// Sets the value for the parameter, returns a 0 status if the new
// value doesn't fall with in the limits.
int ParamItem::SetValue (string val)
{
  (void) val;			// remove compiler warning  
  int error_code = 0;
  return error_code;
}

// Returns the new value in string format.
string ParamItem::GetNewValue ()
{
  string results = name;
  results.append("=");
  return (results);
}

// Returns the new value in string format.
string ParamItem::ComputeIndirString (char *redir)
{
  string    results;
  paramfile par_file;
  if ((par_file = paramopen((char *)file_name.c_str(), NULL, 0, "rH")) != 0)
  {
    char* nval = evaluateIndir(par_file, (char *)name.c_str(), redir);
    if (nval)
    {
      results = nval;
      free(nval);
    }
    paramclose (par_file);
  }

  return (results);
}

// write the parameter info in the parameter file order
void ParamItem::Write (ostream & oss)
{
  oss << name << delim;
  PrintType (oss);
  PrintMode (oss);
}

// Print the parameter type to the output stream
void ParamItem::PrintType (ostream & oss)
{
  switch (type)
    {
    case BOOLEAN_TYPE:
      oss << "b" << delim;
      break;
    case INTEGER_TYPE:
      oss << "i" << delim;
      break;
    case REAL_TYPE:
      oss << "r" << delim;
      break;
    case STRING_TYPE:
      oss << "s" << delim;
      break;
    case FILENAME_TYPE:
      oss << "f" << delim;
      break;
    case PSET_TYPE:
      oss << "pset" << delim;
      break;
    case NO_PARAM_TYPE:	// fallthrough intended 
    default:
      // do nothing 
      break;
    }
}

// Print the parameter mode to the output stream
void ParamItem::PrintMode (ostream & oss)
{
  if ((mode & QUERY_MODE) == QUERY_MODE)
    oss << "q";
  if ((mode & LEARN_MODE) == LEARN_MODE)
    oss << "l";
  if ((mode & HIDDEN_MODE) == HIDDEN_MODE)
    oss << "h";
  if ((mode & AUTO_MODE) == AUTO_MODE)
    oss << "a";
  oss << delim;
}

// Read the parameter from the input stream
void ParamItem::Read (istream & iss)
{
  char temp[MAX_PARAM_LENGTH];

  // Read the name
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  name = temp;

  // Read the type
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if (strchr (temp, 'b'))
    type = BOOLEAN_TYPE;
  else if (strchr (temp, 'i'))
    type = INTEGER_TYPE;
  else if (strchr (temp, 'r'))
    type = REAL_TYPE;
  else if (strstr (temp, "pset"))
    type = PSET_TYPE;
  else if (strchr (temp, 's'))
    type = STRING_TYPE;
  else if (strchr (temp, 'f'))
    type = FILENAME_TYPE;
  else
    type = STRING_TYPE;

  // Read and set the mode
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  mode = CalculateMode (temp);
}

void ParamItem::Read(char *line)
{
  char *where;
  name = where = strtok(line, ",");
  where = strtok (NULL, ",");
  switch (where[0])
    {
    case 'b':
      type = BOOLEAN_TYPE;
      break;
    case 'i':
      type = INTEGER_TYPE;
      break;
    case 'r':
      type = REAL_TYPE;
      break;
    case 'f':
      type = FILENAME_TYPE;
      break;
    case 'p':
      if ((where[1] == 's') && (where[2] == 'e') && (where[3] == 't'))
	type = PSET_TYPE;
      break;
    default:
      type = STRING_TYPE;
      break;
    }
  where = strtok (NULL, ",");
  mode = CalculateMode (where);
}

// ===========================================================================
// Class: ParamString
// 
// Description:
//   Extension of ParamItem class for string type parameters.
// 
// ===========================================================================
// Constructor
ParamString::ParamString ():ParamItem ()
{
  type = STRING_TYPE;
  mode = NO_MODE;
  value.clear ();
  min_value.clear ();
  max_value.clear ();
  current.clear ();
  subset = NULL;
  subset_cnt = 0;
  range_check = NO_RANGE_CHECK;
  cur_redir[0] = val_redir[0] = max_redir[0] = min_redir[0] = '\0';
}

ParamString::ParamString (const ParamItem & item):
ParamItem (item)
{
  value.clear ();
  min_value.clear ();
  max_value.clear ();
  current.clear ();
  subset = NULL;
  subset_cnt = 0;
  range_check = NO_RANGE_CHECK;
  cur_redir[0] = val_redir[0] = max_redir[0] = min_redir[0] = '\0';
  subset_str.clear ();
}

ParamString::ParamString (const ParamString & item):
ParamItem (item)
{
  unsigned int last_char = 0;

  if ((((string) item.value).find ('"') == 0) &&
      (((string) item.value).find_last_of ('"') ==
       (last_char = ((string) item.value).length () - 1)))
    {
      value = ((string) item.value).substr (1, --last_char);
    }
  else
    value = item.value;

  if ((((string) item.max_value).find ('"') == 0) &&
      (((string) item.max_value).find_last_of ('"') ==
       (last_char = ((string) item.max_value).length () - 1)))
    {
      max_value = ((string) item.max_value).substr (1, --last_char);
    }
  else
    max_value = item.max_value;

  if ((((string) item.min_value).find ('"') == 0) &&
      (((string) item.min_value).find_last_of ('"') ==
       (last_char = ((string) item.min_value).length () - 1)))
    {
      min_value = ((string) item.min_value).substr (1, --last_char);
    }
  else
    min_value = item.min_value;

  if ((((string) item.current).find ('"') == 0) &&
      (((string) item.current).find_last_of ('"') ==
       (last_char = ((string) item.current).length () - 1)))
    {
      current = ((string) item.current).substr (1, --last_char);
    }
  else
    current = item.current;

  subset = NULL;
  range_check = item.range_check;
  if ((subset_cnt = item.subset_cnt) != 0)
    {
      subset_str = item.subset_str;
      if (item.subset)
	{
	  if ((subset =
	       (char **) calloc (subset_cnt, sizeof (char *))) != NULL)
	    {
	      int rr, len;

	      for (rr = subset_cnt; rr--;)
		if ((len = strlen (item.subset[rr])) != 0)
		  {
		    if ((subset[rr] =
			 (char *) malloc ((len + 1) * sizeof (char))) == NULL)
		      {
			subset_cnt = 0;
			range_check = NO_RANGE_CHECK;
		      }
		    else
		      strcpy (subset[rr], item.subset[rr]);
		  }
	    }
	  else
	    {
	      subset_cnt = 0;
	      range_check = NO_RANGE_CHECK;
	    }
	}
    }
  strcpy (val_redir, item.val_redir);
  strcpy (cur_redir, item.cur_redir);
  strcpy (max_redir, item.max_redir);
  strcpy (min_redir, item.min_redir);
  if (max_redir[0] != '\0')
    max_value = EvaluateIndir (max_redir);
  if (min_redir[0] != '\0')
    min_value = EvaluateIndir (min_redir);
  file_name = item.file_name;
}

// Destructor
ParamString::~ParamString ()
{
}

string ParamString::EvaluateIndir (char *val)
{
  string
    inds = ComputeIndirString (val);
  return inds;
}

// Sets the value for the parameter, returns a 0 status if the new
// value doesn't fall within the limits.
int ParamString::SetValue (string val)
{
  string compval;
  int status = 1;
  int dquote1, squote1, dquote2, squote2, slen, rr;
  bool indir = false;
  char *temp2 = NULL;

  dquote1 = val.find ('"');
  squote1 = val.find ('\'');
  dquote2 = val.find_last_of ('"');
  squote2 = val.find_last_of ('\'');
  slen = val.length ();

  if (dquote1 != -1)		// at least 1 double quote
    {
      if ((dquote2 == dquote1) && (squote2 == squote1))
	status = 0;		// only 1 double quote- not enclosed in single quotes
    }
  else if (squote1 != -1)	// at least 1 single quote
    {
      if ((dquote2 == dquote1) && (squote2 == squote1))
	status = 0;		// only 1 single quote- not enclosed in double quotes
    }
  if (status != 0)
    {
      status = 0;		// bad unless value is found in subset 

      if ((dquote1 == 0) && (dquote2 == slen - 1))
	compval = val.substr (1, --dquote2);
      else
	compval = val;
      temp2 = (char *) compval.c_str ();
      slen = compval.length ();

      if (((compval == cur_redir) && slen) ||
	  ((compval == current) && (cur_redir[0] == '\0')))
	{
	  // value hasn't changed- avoid  range check on leave cell
	  status = 1;
	  changed = false;
	}
      else
	{
	  if (temp2[0] == ')')
	    {
	      compval = EvaluateIndir (temp2);
	      temp2 = (char *) compval.c_str ();
	      slen = compval.length ();
	      indir = true;
	    }

	  switch (range_check)
	    {
	    case MIN_ONLY_CHECK:
	      if (strcmp (temp2, ((string) min_value).c_str ()) >= 0)
		{
		  current = val;
		  changed = true;
		  status = 1;
		}
	      break;
	    case MAX_ONLY_CHECK:
	      if (strcmp (temp2, ((string) max_value).c_str ()) <= 0)
		{
		  current = val;
		  changed = true;
		  status = 1;
		}
	      break;
	    case SUBSET_CHECK:
	      if ((subset_cnt != 0) && (subset) && (temp2))
		{
		  for (rr = subset_cnt; rr--;)
		    if (strcmp (subset[rr], temp2) == 0)
		      {
			current = val;
			changed = true;
			status = 1;
		      }
		}
	      break;
	    case MIN_MAX_CHECK:
	      if ((strcmp (temp2, ((string) min_value).c_str ()) >= 0) &&
		  (strcmp (temp2, ((string) max_value).c_str ()) <= 0))
		{
		  current = val;
		  changed = true;
		  status = 1;
		}
	      break;
	    case NO_RANGE_CHECK:	// fallthrough intended 
	    case DEFAULT_CHECK:	// fallthrough intended
	    default:
	      current = val;
	      changed = true;
	      status = 1;
	      break;
	    }
	  if (status == 1)
	    {
	      if (indir)
		strcpy (cur_redir, val.c_str ());
	      else
		cur_redir[0] = '\0';
	    }
	}
    }
  return (status);
}

// Returns the new value in string format.
string ParamString::GetNewValue ()
{
  string results = ParamItem::GetNewValue ();

  if (current.find('"') == string::npos)
    results.append("\"");

  results.append(current);

  if (current.find('"') == string::npos)
    results.append("\"");

  return (results);
}

// Print the parameter to the output stream
void ParamString::Print (ostream & oss)
{
  oss << name << delim;

  if (cur_redir[0] == '\0')
    {
      if (current.length () > 0)
	oss << current;
    }
  else
    {
      oss << cur_redir;
      if (current.length () > 0)
	oss << " -> " << current;
    }
  oss << delim;
  if (val_redir[0] == '\0')
    oss << value << delim;
  else
    oss << val_redir << " -> " << value << delim;
  if (prompt.length () > 0)
    oss << prompt;
  oss << delim;
  PrintType (oss);

  switch (range_check)
    {
    case MIN_ONLY_CHECK:
      if (min_redir[0] == '\0')
	oss << min_value << delim;
      else
	oss << min_redir << " -> " << min_value << delim;
      break;
    case MAX_ONLY_CHECK:
      if (max_redir[0] == '\0')
	oss << delim << max_value;
      else
	oss << delim << max_redir << " -> " << max_value;
      break;
    case MIN_MAX_CHECK:
      if (min_redir[0] == '\0')
	oss << min_value << delim;
      else
	oss << min_redir << " -> " << min_value << delim;
      if (max_redir[0] == '\0')
	oss << max_value;
      else
	oss << max_redir << " -> " << max_value;
      break;
    case SUBSET_CHECK:
      if (subset != NULL)
	{
	  int rr;

	  for (rr = 0; rr < subset_cnt; rr++)
	    {
	      oss << subset[rr];
	      if (rr != (subset_cnt - 1))
		oss << "|";
	    }
	  oss << delim;		// don't provide a max for subsets
	}
      break;
    case DEFAULT_CHECK:	// fallthrough intended 
    case NO_RANGE_CHECK:	// fallthrough intended
    default:
      oss << delim;
      break;
    }
  oss << delim;
  PrintMode (oss);
}

// Write the parameter to the output stream
// Read uses strtok which has a problem with <delim><delim>
void ParamString::Write (ostream & oss)
{
  ParamItem::Write (oss);

  if (current.length () > 0)
    oss << current;
  oss << delim;
  if (min_value.length () > 0)
    oss << min_value;
  oss << delim;
  if (max_value.length () > 0)
    oss << max_value;
  oss << delim;
  if (prompt.length () > 0)
    oss << prompt;
  oss << endl;
}

// Read the parameter from the input stream.
// Note: Reads only the value, min_value, max_value, and prompt
void ParamString::Read (istream & iss)
{
  // It is assumed that the name,mode,and type have already been read in.
  int temp_length;
  char temp[MAX_PARAM_LENGTH];
  char buffer[MAX_PARAM_LENGTH];

  if (iss.peek () == '"')
    {
      int length;

      temp[0] = '"';
      iss.getline (buffer, MAX_PARAM_LENGTH, '"');
      iss.getline (&temp[1], MAX_PARAM_LENGTH, '"');
      iss.getline (buffer, MAX_PARAM_LENGTH, ',');
      if ((length = strlen (temp)) < MAX_PARAM_LENGTH)
	{
	  temp[length++] = '"';
	  if (length < MAX_PARAM_LENGTH)
	    temp[length] = '\0';
	}
    }
  else if (iss.peek () == '\'')
    {
      int length;

      temp[0] = '\'';
      iss.getline (buffer, MAX_PARAM_LENGTH, '\'');
      iss.getline (&temp[1], MAX_PARAM_LENGTH, '\'');
      iss.getline (buffer, MAX_PARAM_LENGTH, ',');
      if ((length = strlen (temp)) < MAX_PARAM_LENGTH)
	{
	  temp[length++] = '\'';
	  if (length < MAX_PARAM_LENGTH)
	    temp[length] = '\0';
	}
    }
  else
    {
      iss.getline (temp, MAX_PARAM_LENGTH, ',');
    }
  if (strlen (temp) > 0)
    {
      if (((temp[0] == '"') && (temp[1] == ')')) || (temp[0] == ')'))
	{
	  /* value is a redirect */
	  current = value = '\0';

	  if (temp[0] == '"')
	    {
	      char *last_quote = NULL;

	      /* remove quotes and store */
	      strcpy (val_redir, &temp[1]);
	      last_quote = strrchr (val_redir, '"');
	      if (last_quote)
		*last_quote = '\0';
	    }
	  else
	    strcpy (val_redir, temp);

	  strcpy (cur_redir, val_redir);
	}
      else
	{
	  /* value is not a redirect, update value */
	  value = string (temp);
	  current = value;
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if ((temp_length = strlen (temp)) > 0)
    {
      if (is_string_subset (temp))
	{
	  store_subset (temp);
	  if (subset_cnt != 0)
	    range_check = SUBSET_CHECK;
	}
      else if (temp[0] == ')')
	{
	  min_value = '\0';
	  strcpy (min_redir, temp);
	  range_check = MIN_ONLY_CHECK;
	}
      else
	{
	  int white_space = 1;

	  // don't treat whitespace as a range value 
	  while (temp_length)
	    white_space &= (isspace (temp[--temp_length]) != 0);
	  if (!white_space)
	    {
	      min_value = temp;
	      range_check = MIN_ONLY_CHECK;
	    }
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if ((temp_length = strlen (temp)) > 0)
    {
      if (temp[0] == ')')
	{
	  max_value = '\0';
	  strcpy (max_redir, temp);
	  if (range_check == MIN_ONLY_CHECK)
	    range_check = MIN_MAX_CHECK;
	  else if (range_check == DEFAULT_CHECK)
	    range_check = MAX_ONLY_CHECK;
	}
      else
	{
	  int white_space = 1;

	  // don't treat whitespace as a range value
	  while (temp_length)
	    white_space &= (isspace (temp[--temp_length]) != 0);
	  if (!white_space)
	    {
	      max_value = temp;
	      if (range_check == MIN_ONLY_CHECK)
		range_check = MIN_MAX_CHECK;
	      else if (range_check == NO_RANGE_CHECK)
		range_check = MAX_ONLY_CHECK;
	    }
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, '\0');
  if ((temp_length = strlen (temp)) > 0)
    {
      if ((temp[0] == '"') && (temp[--temp_length] == '"'))
	{
	  temp[temp_length] = '\0';
	  prompt = &temp[1];
	}
    }
}

void ParamString::Read (char *line)
{
  ParamItem::Read (line);

  char *where;
  where = strtok (NULL, ",");
  value = where;
  where = strtok (NULL, ",");
  min_value = where;
  where = strtok (NULL, ",");
  max_value = where;
  where = strtok (NULL, ",");
  prompt = where;
  where = strtok (NULL, ",");
  current = where;		//NOTE: different behavior from other Read()
}

bool ParamString::is_string_subset (char *buffer)
{
  bool ret_val = false;
  if (buffer != NULL)
  {
    int	ii, len, count = 0;

    for (len = ii = strlen (buffer); ii--;)
      if (buffer[ii] == '|')
	count++;

    // subset must contain at least 1 '|' but not as first/last char
    ret_val |= (count && (buffer[0] != '|') && (buffer[len] != '|'));
  }

  return (ret_val);
}

void ParamString::store_subset (char *buffer)
{
  if (buffer != NULL)
    {
      int len;
      char *tp, *temp_buffer = NULL;
      char set[] = "|";

      len = strlen (buffer);
      if ((temp_buffer = (char *) malloc (len * sizeof (char) + 1)) != NULL)
	{
	  int ii, cnt = 0, quote_cnt = 0;

	  // get number of elements in subset
	  subset_cnt = 1;
	  for (ii = strlen (buffer); ii--;)
	    {
	      if (buffer[ii] == '|')
		subset_cnt++;
	      if (buffer[ii] == '"')
		quote_cnt++;
	    }

	  // copy buffer since strtok destructively modifies string
	  if (quote_cnt == 2 && buffer[0] == '"' && buffer[len - 1] == '"')
	    {
	      strcpy (temp_buffer, &buffer[1]);
	      temp_buffer[len - 2] = '\0';
	    }
	  else
	    strcpy (temp_buffer, buffer);
	  subset_str = temp_buffer;

	  if (subset_cnt > 1)	// allocate memory to store subset
	    subset = (char **) malloc ((subset_cnt + 1) * sizeof (char *));

	  // store first element of subset
	  if ((tp = strtok (temp_buffer, set)) != NULL)
	    {
	      subset[cnt] =
		(char *) malloc ((strlen (tp) + 1) * sizeof (char));
	      strcpy (subset[cnt++], tp);
	    }

	  // go through rest of list
	  while ((tp = strtok (NULL, set)) && (cnt < subset_cnt))
	    {
	      subset[cnt] =
		(char *) malloc ((strlen (tp) + 1) * sizeof (char));
	      strcpy (subset[cnt++], tp);
	    }

	  // free memory
	  if (temp_buffer)
	    {
	      free (temp_buffer);
	      temp_buffer = NULL;
	    }
	}
    }				// end if {buffer != NULL}
}

// ===========================================================================
// Class: ParamInteger
// 
// Description:
//   Extension of ParamItem class for integer type parameters.
// 
// ===========================================================================
// Constructor
ParamInteger::ParamInteger ():ParamItem ()
{
  type = INTEGER_TYPE;
  mode = NO_MODE;
  value = current = 0;
  min_value = INT_MIN;
  max_value = INT_MAX;
  subset = NULL;
  subset_cnt = 0;
  range_check = NO_RANGE_CHECK;
  cur_redir[0] = val_redir[0] = max_redir[0] = min_redir[0] = '\0';
  subset_str.clear ();
}

ParamInteger::ParamInteger (const ParamItem & item):ParamItem (item)
{
  value = current = 0;
  min_value = INT_MIN;
  max_value = INT_MAX;
  subset = NULL;
  subset_cnt = 0;
  range_check = NO_RANGE_CHECK;
  cur_redir[0] = val_redir[0] = max_redir[0] = min_redir[0] = '\0';
  subset_str.clear ();
}

ParamInteger::ParamInteger (const ParamInteger & item):ParamItem (item)
{
  value = item.value;
  mode = item.mode;
  min_value = item.min_value;
  max_value = item.max_value;
  current = item.current;
  subset = NULL;
  range_check = item.range_check;
  if ((subset_cnt = item.subset_cnt) != 0)
    {
      subset_str = item.subset_str;
      if (item.subset)
	{
	  if ((subset = (int *) calloc (subset_cnt, sizeof (int))) != NULL)
	    {
	      int rr;
	      for (rr = subset_cnt; rr--;)
		subset[rr] = item.subset[rr];
	    }
	  else
	    {
	      subset_cnt = 0;
	      range_check = NO_RANGE_CHECK;
	    }
	}
    }
  strcpy (cur_redir, item.cur_redir);
  strcpy (val_redir, item.val_redir);
  strcpy (max_redir, item.max_redir);
  strcpy (min_redir, item.min_redir);
  if (max_redir[0] != '\0')
    max_value = EvaluateIndir (max_redir);
  if (min_redir[0] != '\0')
    min_value = EvaluateIndir (min_redir);
  file_name = item.file_name;
}

// Destructor
ParamInteger::~ParamInteger ()
{
}

int ParamInteger::EvaluateIndir (char *val)
{
  int rval = INDEF_INT;
  string inds = ComputeIndirString(val);

  if (inds.length ())
  {
    long lv;
    if (_cxcparam_string_to_long(inds, lv) == 0)
      rval = lv;
  }
  return rval;
}

// Sets the value for the parameter, returns a 0 status if the new
// value doesn't fall with in the limits.
int ParamInteger::SetValue(int val)
{
  int error = 0;

  if (val == INDEF_INT)
  {
    changed = (val != current);
    current = INDEF_INT;
    error = 1;
  }
  else
  {
    switch (range_check)
    {
    case MIN_ONLY_CHECK:
      if ((min_value == INDEF_INT) || (val >= min_value))
      {
	if (current != val)
	{
	  current = val;
	  changed = true;
	}
	error = 1;
      }
      break;
    case MAX_ONLY_CHECK:
      if ((max_value == INDEF_INT) || (val <= max_value))
      {
	if (current != val)
	{
	  current = val;
	  changed = true;
	}
	error = 1;
      }
      break;
    case SUBSET_CHECK:
      if ((subset_cnt != 0) && (subset))
      {
	int rr;
	for (rr = subset_cnt; rr--;)
	{
	  if (subset[rr] == val)
	  {
	    if (current != val)
	    {
	      current = val;
	      changed = true;
	    }
	    error = 1;
	  }
	}
      }
      break;
    case NO_RANGE_CHECK:// fallthrough intended
    case MIN_MAX_CHECK:	// fallthrough intended 
    case DEFAULT_CHECK:	// fallthrough intended 
    default:
      if ((min_value == INDEF_INT) || (min_value == INDEF_INT) || (min_value <= max_value))
      {
	// a param w/ at least 1 legal value
	if (((min_value == INDEF_INT) || (val >= min_value)) &&
	    ((max_value == INDEF_INT) || (val <= max_value)))
	{
	  if (current != val)
	  {
	    current = val;
	    changed = true;
	  }
	  error = 1;
	}
	else if (current == val)
	{
	  /* the value is not being changed from what it was- but
	   * since it is possible to get into an infinite loop of
	   * error dialogs from ParamMatrix::LeaveCell (it checks
	   * the data, which may already be invalid), allow the 
	   * value to remain unchanged. 
	   */
	  error = 1;
	  changed = false;
	}
      }
      break;
    }
  }
  return (error);
}

int ParamInteger::SetValue(string val)
{
  int error = 0;
  long lv;

  if ( _cxcparam_strcmp_loose(val, "indef") == 0 )
  {
    error = SetValue(INDEF_INT);
    cur_redir[0] = '\0';
  }
  else if ( _cxcparam_string_to_long(val, lv) == 0 )
  {
    if ((error = SetValue((int) lv)) == 1)
      cur_redir[0] = '\0';
  }
  else if ( ((string)val).find(')') == 0 )
  {
    if (val != (const char *)cur_redir)
    {
      int tval = EvaluateIndir((char *)val.c_str());
      if ((error = SetValue(tval)) == 1)
	strcpy(cur_redir, val.c_str());
    }
    else
    {
      error = 1;
      changed = false;
    }
  }
  return (error);
}

// Returns the new value in string format.
string ParamInteger::GetNewValue()
{
  string results = ParamItem::GetNewValue();
  ostringstream oss;

  if (current == INDEF_INT)
    oss << "INDEF";
  else
    oss << current;

  results.append( oss.str() );

  return (results);
}

// Print the parameter to the output stream
//   Format: <name>,<current>,<value>,<prompt>,<type>,[<min>],[<max>],<mode>
void ParamInteger::Print(ostream &oss)
{
  oss << name << delim;
  if (cur_redir[0] == '\0')
    {
      if (current == INDEF_INT)
	oss << "INDEF" << delim;
      else
	oss << current << delim;
    }
  else
    {
      if (current == INDEF_INT)
	oss << cur_redir << " -> INDEF" << delim;
      else
	oss << cur_redir << " -> " << current << delim;
    }

  if (val_redir[0] == '\0')
    {
      if (value == INDEF_INT)
	oss << "INDEF" << delim;
      else
	oss << value << delim;
    }
  else
    {
      if (value == INDEF_INT)
	oss << val_redir << " -> INDEF" << delim;
      else
	oss << val_redir << " -> " << value << delim;
    }
  oss << prompt << delim;
  PrintType (oss);
  switch (range_check)
    {
    case MIN_ONLY_CHECK:
      if (min_redir[0] == '\0')
	{
	  if (min_value == INDEF_INT)
	    oss << "INDEF" << delim;
	  else
	    oss << min_value << delim;
	}
      else
	{
	  if (min_value == INDEF_INT)
	    oss << min_redir << " -> " << "INDEF" << delim;
	  else
	    oss << min_redir << " -> " << min_value << delim;
	}
      break;
    case MAX_ONLY_CHECK:
      if (max_redir[0] == '\0')
	{
	  if (max_value == INDEF_INT)
	    oss << delim << "INDEF";
	  else
	    oss << delim << max_value;
	}
      else
	{
	  if (max_value == INDEF_INT)
	    oss << delim << max_redir << " -> INDEF";
	  else
	    oss << delim << max_redir << " -> " << max_value;
	}
      break;
    case MIN_MAX_CHECK:
      if (min_redir[0] == '\0')
	{
	  if (min_value == INDEF_INT)
	    oss << "INDEF" << delim;
	  else
	    oss << min_value << delim;
	}
      else
	{
	  if (min_value == INDEF_INT)
	    oss << min_redir << " -> INDEF" << delim;
	  else
	    oss << min_redir << " -> " << min_value << delim;
	}
      if (max_redir[0] == '\0')
	{
	  if (max_value == INDEF_INT)
	    oss << "INDEF";
	  else
	    oss << max_value;
	}
      else
	{
	  if (max_value == INDEF_INT)
	    oss << max_redir << " -> INDEF";
	  else
	    oss << max_redir << " -> " << max_value;
	}
      break;
    case SUBSET_CHECK:
      if (subset != NULL)
	{
	  int rr;

	  for (rr = 0; rr < subset_cnt; rr++)
	    {
	      oss << subset[rr];
	      if (rr != (subset_cnt - 1))
		oss << "|";
	    }
	  oss << delim;		// don't provide a max for subsets 
	}
      break;
    case DEFAULT_CHECK:	// fallthrough intended 
    case NO_RANGE_CHECK:	// fallthrough intended 
    default:
      oss << delim;
      break;
    }
  oss << delim;
  PrintMode (oss);
}

// Write the parameter to the output stream
void ParamInteger::Write (ostream &oss)
{
  ParamItem::Write(oss);

  if (current == INDEF_INT)
    oss << "INDEF" << delim;
  else
    oss << current << delim;

  if (min_value == INDEF_INT)
    oss << "INDEF" << delim;
  else
    oss << min_value << delim;

  if (max_value == INDEF_INT)
    oss << "INDEF" << delim;
  else
    oss << max_value << delim;

  oss << prompt << endl;
}

// Read the parameter from the input stream.
// Note: Reads only the value, min_value, max_value, and prompt
void ParamInteger::Read(istream &iss)
{
  // It is assumed that the name,mode,and type have already been read in.
  char temp[MAX_PARAM_LENGTH];

  iss.getline(temp, MAX_PARAM_LENGTH, ',');
  if (strlen (temp) > 0)
  {
    if (((temp[0] == '"') && (temp[1] == ')')) || (temp[0] == ')'))
    {
      current = value = 0;
      if (temp[0] == '"')
      {
	char *last_quote = NULL;

	/* remove quotes and store */
	strcpy (val_redir, &temp[1]);
	last_quote = strrchr (val_redir, '"');
	if (last_quote)
	  *last_quote = '\0';
      }
      else
	strcpy (val_redir, temp);
      strcpy (cur_redir, val_redir);
    }
    else if ((strcmp (temp, "INDEF") == 0) || (strcmp (temp, "indef") == 0))
    {
      current = value = INDEF_INT;
    }
    else
    {
      value = atoi (temp);
      current = value;
    }
  }
  iss.getline(temp, MAX_PARAM_LENGTH, ',');
  if (strlen (temp) > 0)
  {
    if (is_numeric_subset (temp))
    {
      store_subset (temp);
      if (subset_cnt != 0)
	range_check = SUBSET_CHECK;
    }
    else if (is_numeric (temp))
    {
      min_value = atoi (temp);
      range_check = MIN_ONLY_CHECK;
    }
    else if (temp[0] == ')')
    {
      strcpy (min_redir, temp);
      min_value = EvaluateIndir (min_redir);
      range_check = MIN_ONLY_CHECK;
    }
  }
  iss.getline(temp, MAX_PARAM_LENGTH, ',');
  if (strlen(temp) > 0)
  {
    if (is_numeric(temp))
    {
      max_value = atoi(temp);
      if (range_check == MIN_ONLY_CHECK)
	range_check = MIN_MAX_CHECK;
      else if (range_check == NO_RANGE_CHECK)
	range_check = MAX_ONLY_CHECK;
    }
    else if (temp[0] == ')')
    {
      strcpy (max_redir, temp);
      max_value = EvaluateIndir (max_redir);
      if (range_check == MIN_ONLY_CHECK)
	range_check = MIN_MAX_CHECK;
      else if (range_check == NO_RANGE_CHECK)
	range_check = MAX_ONLY_CHECK;
    }
  }
  iss.getline(temp, MAX_PARAM_LENGTH, '\0');
  int temp_length;
  if ((temp_length = strlen(temp)) > 0)
  {
    if ((temp[0] == '"') && (temp[--temp_length] == '"'))
    {
      temp[temp_length] = '\0';
      prompt = &temp[1];
    }
  }
}

void ParamInteger::Read(char *line)
{
  // Read base fields
  ParamItem::Read(line);

  // Read type dependent fields and propmt
  char* where;
  where = strtok(NULL, ",");
  value = atoi(where);

  where = strtok(NULL, ",");
  min_value = atoi(where);

  where = strtok(NULL, ",");
  max_value = atoi(where);

  where = strtok(NULL, ",");
  prompt = where;

  // NOTE: different behavior from other Read()
  //       ?? current value follows prompt ??
  where = strtok(NULL, ",");
  if ( where )
    current = atoi(where);
}

// ---------------------------------------------------------
// Check if character string represents an integer 
//  ie. composed of: numbers, '-', or '+'
// 
// NOTE.. this super liberal ( '7+3' == OK )
//
// returns
//   false = NULL buffer 
//           string contains chars other than expected
//   true  = if string matches expectations
// ---------------------------------------------------------
bool ParamInteger::is_numeric (char *buffer)
{
  bool ret_val = true;
  if (buffer != NULL)
  {
    int	ii;
    for (ii = strlen(buffer); ii-- ; )
      ret_val &= (( isdigit(buffer[ii]) != 0) || (buffer[ii] == '-') || (buffer[ii] == '+') );
  }
  else
    ret_val = false;

  return (ret_val);
}

// check if a string contains a subset of | delineated integers 
bool ParamInteger::is_numeric_subset (char *buffer)
{
  bool ret_val = true;
  if (buffer != NULL)
  {
    int	ii, len, count = 0;

    for (len = ii = strlen(buffer); ii-- ; )
    {
      ret_val &= ((isdigit (buffer[ii]) != 0) || (buffer[ii] == '-') || (buffer[ii] == '+') || (buffer[ii] == '|'));
      if (buffer[ii] == '|')
	count++;
    }
    // subset must contain at least 1 '|' but not as first/last char
    // MCD NOTE: BUG.. should be checking [len-1].. not catching end case
    ret_val &= (count && (buffer[0] != '|') && (buffer[len] != '|'));
  }
  else
    ret_val = false;

  return (ret_val);
}

void ParamInteger::store_subset (char *buffer)
{
  if (buffer != NULL)
  {
    int ii, len;
    char *temp_buffer = NULL;
    char *tp;
    char set[] = "|";

    subset_str = buffer;
    len = strlen (buffer);
    if ((temp_buffer = (char *)malloc(len * sizeof(char) + 1)) != NULL )
    {
      int cnt = 0;

      // copy buffer since strtok destructively modifies string 
      strcpy(temp_buffer, buffer);

      // get number of elements in subset 
      subset_cnt = 1;
      for (ii = strlen (buffer); ii--;)
	if (buffer[ii] == '|')
	  subset_cnt++;
      if (subset_cnt > 1)	// allocate memory to store subset 
	subset = (int *) calloc(subset_cnt, sizeof (int));

      // store first element of subset  
      tp = strtok(temp_buffer, set);
      if (is_numeric(tp))
	subset[cnt++] = atoi(tp);

      // go through rest of list 
      while ((tp = strtok(NULL, set)) && (cnt < subset_cnt))
	if (is_numeric(tp))
	  subset[cnt++] = atoi(tp);

      // free memory 
      if (temp_buffer)
      {
	free (temp_buffer);
	temp_buffer = NULL;
      }
    }
  } // end if {buffer != NULL}  
}

// ===========================================================================
// Class: ParamReal
// 
// Description:
//   Extension of ParamItem class for real type parameters.
// 
// ===========================================================================

// Constructor
ParamReal::ParamReal():ParamItem()
{
  type = REAL_TYPE;
  mode = NO_MODE;
  value = current = 0.0;
  min_value = DBL_MAX * -1.0;
  max_value = DBL_MAX;
  subset = NULL;
  subset_cnt = 0;
  range_check = NO_RANGE_CHECK;
  cur_redir[0] = val_redir[0] = max_redir[0] = min_redir[0] = '\0';
  precision = 6;
  subset_str.clear ();
}

ParamReal::ParamReal(const ParamItem & item):ParamItem(item)
{
  value = current = 0.0;
  min_value = DBL_MAX * -1.0;
  max_value = DBL_MAX;
  subset = NULL;
  subset_cnt = 0;
  range_check = NO_RANGE_CHECK;
  cur_redir[0] = val_redir[0] = max_redir[0] = min_redir[0] = '\0';
  precision = 6;
  subset_str.clear ();
}

ParamReal::ParamReal(const ParamReal &item):ParamItem (item)
{
  value = item.value;
  mode = item.mode;
  min_value = item.min_value;
  max_value = item.max_value;
  current = item.current;
  precision = item.precision;
  subset = NULL;
  range_check = item.range_check;
  if ((subset_cnt = item.subset_cnt) != 0)
    {
      subset_str = item.subset_str;
      if (item.subset)
	{
	  if ((subset =
	       (double *) calloc (subset_cnt, sizeof (double))) == NULL)
	    {
	      subset_cnt = 0;
	      range_check = NO_RANGE_CHECK;
	    }
	  else
	    for (int rr = subset_cnt; rr--;)
	      subset[rr] = item.subset[rr];
	}
    }
  strcpy (cur_redir, item.cur_redir);
  strcpy (val_redir, item.val_redir);
  strcpy (max_redir, item.max_redir);
  strcpy (min_redir, item.min_redir);
  if (max_redir[0] != '\0')
    max_value = EvaluateIndir (max_redir);
  if (min_redir[0] != '\0')
    min_value = EvaluateIndir (min_redir);
  file_name = item.file_name;
}

// set precision for 'Reals' - return previous value
int ParamReal::SetPrecision (int val)
{
  int orig = precision;
  precision = val;
  return orig;
}

// Destructor
ParamReal::~ParamReal ()
{
}

double ParamReal::EvaluateIndir (char *val)
{
  double dd, rval = INDEF_REAL;

  string inds = ComputeIndirString (val);
  if (inds.length ())
    if (!_cxcparam_string_to_double (inds, dd))
      rval = dd;

  return rval;
}


// Sets the value for the parameter, returns a 0 status if the new
// value doesn't fall with in the limits.
int ParamReal::SetValue(double val)
{
  int error = 0;

  if (val == INDEF_REAL)
  {
    changed = (val != current);
    current = INDEF_REAL;
    error = 1;
  }
  else
  {
    switch (range_check)
    {
    case MIN_ONLY_CHECK:
      if ((min_value == INDEF_REAL) || (val >= min_value))
      {
	if (current != val)
	{
	  current = val;
	  changed = true;
	}
	error = 1;
      }
      break;
    case MAX_ONLY_CHECK:
      if ((max_value == INDEF_REAL) || (val <= max_value))
      {
	if (current != val)
	{
	  current = val;
	  changed = true;
	}
	error = 1;
      }
      break;
    case SUBSET_CHECK:
      if ((subset_cnt != 0) && (subset))
      {
	int rr;
	
	for (rr = subset_cnt; rr--;)
	{
	  if (subset[rr] == val)	// should check +/- FLT_EPSILON 
	  {
	    if (current != val)
	    {
	      current = val;
	      changed = true;
	    }
	    error = 1;
	  }
	}
      }
      break;
    case NO_RANGE_CHECK:
      if (current != val)
      {
	current = val;
	changed = true;
      }
      error = 1;
      break;
    case MIN_MAX_CHECK:	// fallthrough intended
    case DEFAULT_CHECK:	// fallthrough intended
    default:
      if ((min_value == INDEF_REAL) || (min_value == INDEF_REAL) || (min_value <= max_value))	// a param w/ at least 1 legal value
      {
	if (((min_value == INDEF_REAL) || (val >= min_value)) &&
	    ((max_value == INDEF_REAL) || (val <= max_value)))
	{
	  if (current != val)
	  {
	    current = val;
	    changed = true;
	  }
	  error = 1;
	}
	else if (current == val)
	{
	  /* the value is not being changed from what it was- but
	   * since it is possible to get into an infinite loop of
	   * error dialogs from ParamMatrix::LeaveCell (it checks
	   * the data, which may already be invalid), so allow the 
	   * value to remain unchanged
	   */
	  error = 1;
	  changed = false;
	}
      }
      break;
    }			/* end switch */
  }
  return (error);
}

int ParamReal::SetValue (string val)
{
  int error = 0;

  if (((string) val).find (')') == 0)
  {
    if (val != (const char *) cur_redir)
    {
      double tval = EvaluateIndir((char *)val.c_str());
      error = SetValue(tval);
      if (error == 1)
      {
	strcpy(cur_redir, val.c_str());
	if (tval != INDEF_REAL)
	{
	  string inds = ComputeIndirString((char *) val.c_str());
	  int ln = inds.length();
	  int pos = inds.find('.');
	  double dd;
	  if (!_cxcparam_string_to_double(inds, dd) && (pos != -1) &&
	      (ln > pos))
	    SetPrecision(ln - pos);
	}
      }
    }
    else
    {
      error = 1;
      changed = false;
    }
  }
  else if (_cxcparam_strcmp_loose(val, "indef") == 0)
  {
    error = SetValue (INDEF_REAL);
    cur_redir[0] = '\0';
  }
  else
  {
    double temp_d, temp_f;
    char sptr[256];
    char *ptr = NULL;

    /* assumes that its not a redirect */
    strcpy(sptr, ((string) val).c_str());
    temp_d = strtod(sptr, &ptr);
    if (ptr[0] == '\0')
    {
      // use basic_istringstream to avoid precisions issues w/ cxcparam lib 
      istringstream(sptr) >> temp_f;
      if (strchr(sptr, '.'))
	SetPrecision(strlen(sptr));

      /* if previous value was a redirect- flag to write update */
      if ((error = SetValue(temp_f)) == 1)
	cur_redir[0] = '\0';
    }
  }
  return (error);
}

// Returns the new value in string format.
string ParamReal::GetNewValue()
{
  string results = ParamItem::GetNewValue();
  ostringstream oss;

  if (current == INDEF_REAL)
    oss << "INDEF";
  else
    oss << setprecision(precision) << current;

  results.append( oss.str() );
  return (results);
}

// Print the parameter to the output stream
void ParamReal::Print (ostream & oss)
{
  oss << name << delim;
  if (cur_redir[0] == '\0')
  {
    if (current == INDEF_REAL)
      oss << "INDEF" << delim;
    else
      oss << setprecision (precision) << current << delim;
  }
  else
  {
    if (current == INDEF_REAL)
      oss << cur_redir << " -> INDEF" << delim;
    else
      oss << cur_redir << " -> " << setprecision (precision) << current <<
	delim;
  }

  if (val_redir[0] == '\0')
  {
    if (value == INDEF_REAL)
      oss << "INDEF" << delim << prompt << delim;
    else
      oss << setprecision (precision) << value << delim << prompt << delim;
  }
  else
  {
    if (value == INDEF_REAL)
      oss << val_redir << " -> INDEF" << delim << prompt << delim;
    else
      oss << val_redir << " -> " << setprecision (precision) << value <<
	delim << prompt << delim;
  }
  PrintType (oss);
  switch (range_check)
  {
    case MIN_ONLY_CHECK:
      if (min_redir[0] == '\0')
      {
	if (min_value == INDEF_REAL)
	  oss << "INDEF" << delim;
	else
	  oss << min_value << delim;
      }
      else
      {
	if (min_value == INDEF_REAL)
	  oss << min_redir << " -> " << "INDEF" << delim;
	else
	  oss << min_redir << " -> " << min_value << delim;
      }
      break;
    case MAX_ONLY_CHECK:
      if (max_redir[0] == '\0')
      {
	if (max_value == INDEF_REAL)
	  oss << delim << "INDEF";
	else
	  oss << delim << max_value;
      }
      else
      {
	if (max_value == INDEF_REAL)
	  oss << delim << max_redir << " -> " << "INDEF";
	else
	  oss << delim << max_redir << " -> " << max_value;
      }
      break;
    case MIN_MAX_CHECK:
      if (min_redir[0] == '\0')
      {
	if (min_value == INDEF_REAL)
	  oss << "INDEF" << delim;
	else
	  oss << min_value << delim;
      }
      else
      {
	if (min_value == INDEF_REAL)
	  oss << min_redir << " -> " << "INDEF" << delim;
	else
	  oss << min_redir << " -> " << min_value << delim;
      }
      if (max_redir[0] == '\0')
      {
	if (max_value == INDEF_REAL)
	  oss << "INDEF";
	else
	  oss << max_value;
      }
      else
      {
	if (max_value == INDEF_REAL)
	  oss << max_redir << " -> " << "INDEF";
	else
	  oss << max_redir << " -> " << max_value;
      }
      break;
    case SUBSET_CHECK:
      if (subset != NULL)
      {
	int rr;
	for (rr = 0; rr < subset_cnt; rr++)
	{
	  oss << subset[rr];
	  if (rr != (subset_cnt - 1))
	    oss << "|";
	}
	oss << delim;		// don't provide a max for subsets
      }
      break;
  case DEFAULT_CHECK:	// fallthrough intended 
  case NO_RANGE_CHECK:	// fallthrough intended
  default:
    oss << delim;
    break;
  }
  oss << delim;
  PrintMode (oss);
}

// Write the parameter to the output stream
void ParamReal::Write (ostream &oss)
{
  // Write base fields
  ParamItem::Write(oss);

  // Write current value, min, and max.
  if (current == INDEF_REAL)
    oss << "INDEF" << delim;
  else
    oss << setprecision (precision) << current << delim;

  if (min_value == INDEF_REAL)
    oss << "INDEF" << delim;
  else
    oss << min_value << delim;

  if (max_value == INDEF_REAL)
    oss << "INDEF" << delim;
  else
    oss << max_value << delim;
}

// Read the parameter from the input stream.
// Note: Reads only the value, min_value, max_value, and prompt
void ParamReal::Read (istream & iss)
{
  // It is assumed that the name,mode,and type have already been read in.
  char temp[MAX_PARAM_LENGTH];

  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if (strlen (temp) > 0)
    {
      if (((temp[0] == '"') && (temp[1] == ')')) || (temp[0] == ')'))
	{
	  current = value = 0.0;
	  if (temp[0] == '"')
	    {
	      char *last_quote = NULL;

	      /* remove quotes and store */
	      strcpy (val_redir, &temp[1]);
	      last_quote = strrchr (val_redir, '"');
	      if (last_quote)
		*last_quote = '\0';
	    }
	  else
	    strcpy (val_redir, temp);

	  strcpy (cur_redir, val_redir);
	}
      else if ((strcmp (temp, "INDEF") == 0) || (strcmp (temp, "indef") == 0))
	{
	  current = value = INDEF_REAL;
	}
      else
	{
	  istringstream (temp) >> value;
	  current = value;
	  string stemp = temp;
	  if (stemp.find ('.') != string::npos)
	    SetPrecision (stemp.length ());
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if (strlen (temp) > 0)
    {
      if (is_decimal_subset (temp))
	{
	  store_subset (temp);
	  if (subset_cnt != 0)
	    range_check = SUBSET_CHECK;
	}
      else if (is_decimal_number (temp))
	{
	  min_value = atof (temp);
	  range_check = MIN_ONLY_CHECK;
	}
      else if (temp[0] == ')')
	{
	  min_value = 0;
	  strcpy (min_redir, temp);
	  range_check = MIN_ONLY_CHECK;
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if (strlen (temp) > 0)
    {
      if (is_decimal_number (temp))
	{
	  max_value = atof (temp);
	  if (range_check == MIN_ONLY_CHECK)
	    range_check = MIN_MAX_CHECK;
	  else if (range_check == NO_RANGE_CHECK)
	    range_check = MAX_ONLY_CHECK;
	}
      else if (temp[0] == ')')
	{
	  max_value = 0;
	  strcpy (max_redir, temp);
	  if (range_check == MIN_ONLY_CHECK)
	    range_check = MIN_MAX_CHECK;
	  else if (range_check == NO_RANGE_CHECK)
	    range_check = MAX_ONLY_CHECK;
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, '\0');
  int temp_length;
  if ((temp_length = strlen (temp)) > 0)
    {
      if ((temp[0] == '"') && (temp[--temp_length] == '"'))
	{
	  temp[temp_length] = '\0';
	  prompt = &temp[1];
	}
    }
}

void ParamReal::Read (char* line)
{
  // Read base fields
  ParamItem::Read (line);

  // Read type dependent fields and propmt
  char* where;
  where = strtok(NULL, ",");
  value = atof(where);

  string temp = where;
  if (temp.find('.') != string::npos)
    SetPrecision( temp.length() );

  where = strtok(NULL, ",");
  min_value = atof(where);

  where = strtok(NULL, ",");
  max_value = atof(where);

  where = strtok(NULL, ",");
  prompt = where;

  // NOTE: different behavior from other Read()
  //       ?? current value follows prompt ??
  where = strtok(NULL, ",");
  if ( where )
    current = atof(where);
}

// Checks if a string represents a real number 
//  ie. sequence of <digit>,'-','+','.',
bool ParamReal::is_decimal_number(char *buffer)
{
  bool ret_val = true;
  if (buffer != NULL)
  {
    int ii;
    for (ii = strlen(buffer); ii--;)
      ret_val &= ((isdigit(buffer[ii]) != 0) || (buffer[ii] == '-') ||
		  (buffer[ii] == '+') || (buffer[ii] == '.'));
  }
  else
    ret_val = false;

  return (ret_val);
}

// checks if a string contains a | delineated real subset 
bool ParamReal::is_decimal_subset (char *buffer)
{
  bool ret_val = true;

  if (buffer != NULL)
  {
    int ii, len, count = 0;
    for (len = ii = strlen (buffer); ii--;)
    {
      ret_val &= ((isdigit (buffer[ii]) != 0) || (buffer[ii] == '-') ||
		  (buffer[ii] == '+') || (buffer[ii] == '|') ||
		  (buffer[ii] == '.'));
      if (buffer[ii] == '|')
	count++;
    }

    // subset must contain at least 1 '|' but not as first/last char
    ret_val &= (count && (buffer[0] != '|') && (buffer[len] != '|'));
  }
  else
    ret_val = false;

  return (ret_val);
}

void ParamReal::store_subset (char *buffer)
{
  if (buffer != NULL)
  {
    int ii, len;
    char *tp, *temp_buffer = NULL;
    char set[] = "|";

    subset_str = buffer;
    len = strlen (buffer);
    if ((temp_buffer = (char *)malloc (len * sizeof(char) + 1)) != NULL)
    {
      int cnt = 0;

      // copy buffer since strtok destructively modifies string
      strcpy(temp_buffer, buffer);

      // get number of elements in subset
      subset_cnt = 1;
      for (ii = strlen (buffer); ii--;)
	if (buffer[ii] == '|')
	  subset_cnt++;
      if (subset_cnt > 1)	// allocate memory to store subset
	subset = (double *)calloc(subset_cnt, sizeof(double));

      // store first element of subset
      tp = strtok(temp_buffer, set);
      if (is_decimal_number(tp))
	subset[cnt++] = atof(tp);

      // go through rest of list
      while ((tp = strtok(NULL, set)) && (cnt < subset_cnt))
	if (is_decimal_number(tp))
	  subset[cnt++] = atof(tp);

      // free memory
      if (temp_buffer)
      {
	free(temp_buffer);
	temp_buffer = NULL;
      }
    }
  } // end if {buffer != NULL}
}

// ===========================================================================
// Class: ParamBoolean
// 
// Description:
//   Extension of ParamItem class for boolean type parameters.
// 
// ===========================================================================

// Constructor
ParamBoolean::ParamBoolean ():ParamItem ()
{
  type = BOOLEAN_TYPE;
  value = current = false;
  min_value = false;
  max_value = true;
  cur_redir[0] = val_redir[0] = '\0';
}

ParamBoolean::ParamBoolean (const ParamItem & item):
ParamItem (item)
{
  value = current = false;
  min_value = false;
  max_value = true;
  cur_redir[0] = val_redir[0] = '\0';
}

ParamBoolean::ParamBoolean (const ParamBoolean & item):
ParamItem (item)
{
  value = item.value;
  mode = item.mode;
  min_value = false;
  max_value = true;
  current = item.current;
  strcpy (val_redir, item.val_redir);
  strcpy (cur_redir, item.cur_redir);
}

// Destructor
ParamBoolean::~ParamBoolean ()
{
}

bool ParamBoolean::EvaluateIndir (char *val)
{
  bool
    rval = false;

  string
    inds = ComputeIndirString (val);
  if (inds.length ())
    {
      if ((_cxcparam_strcmp_loose (val, "yes") == 0) ||
	  (_cxcparam_strcmp_loose (val, "true") == 0) ||
	  (_cxcparam_strcmp_loose (val, "1") == 0))
	rval = true;
    }

  return rval;
}


// Sets the value for the parameter, returns a 0 status if the new
// value doesn't fall with in the limits.
int ParamBoolean::SetValue (bool val)
{
  if (current != val)
    {
      current = val;
      changed = true;
    }
  return (1);
}

int ParamBoolean::SetValue (string val)
{
  int error = 0;

  if (((string) val).find (')') == 0)
    {
      if (val != (const char *) cur_redir)
	{
	  bool tval = EvaluateIndir ((char *) val.c_str ());
	  if ((error = SetValue (tval)) == 1)
	    strcpy (cur_redir, val.c_str ());
	}
      else
	{
	  error = 1;
	  changed = false;
	}
    }
  else if (val == "yes")
    {
      if (current != true)
	current = changed = true;
      error = 1;
      if (cur_redir[0] != '\0')
	{
	  cur_redir[0] = '\0';
	  current = changed = true;
	}
    }
  else if (val == "no")
    {
      if (current != false)
	{
	  current = false;
	  changed = true;
	}
      error = 1;
      if (cur_redir[0] != '\0')
	{
	  current = false;
	  cur_redir[0] = '\0';
	  changed = true;
	}
    }

  return (error);
}

// Returns the new value in string format.
string ParamBoolean::GetNewValue ()
{
  string
    results = ParamItem::GetNewValue ();
  if (current)
    results.append ("yes");
  else
    results.append ("no");
  return (results);
}

// Print the parameter to the output stream
void ParamBoolean::Print (ostream & oss)
{
  ParamItem::Print (oss);
  oss << name << delim;
  if (cur_redir[0] != '\0')
    oss << cur_redir << " -> ";
  if (current)
    oss << "yes" << delim;
  else
    oss << "no" << delim;
  if (val_redir[0] != '\0')
    oss << val_redir << " -> ";
  if (value)
    oss << "yes" << delim;
  else
    oss << "no" << delim;
  oss << prompt << delim;
  PrintType (oss);
  if (min_value)
    oss << "yes" << delim;
  else
    oss << "no" << delim;
  if (max_value)
    oss << "yes";
  else
    oss << "no";
  oss << delim;
  PrintMode (oss);
}

// Write the parameter to the output stream
void ParamBoolean::Write (ostream & oss)
{
  ParamItem::Write (oss);
  if (current)
    oss << "yes" << delim;
  else
    oss << "no" << delim;
  if (min_value)
    oss << "yes" << delim;
  else
    oss << "no" << delim;
  if (max_value)
    oss << "yes" << delim;
  else
    oss << "no" << delim;
  oss << prompt << endl;
}

// Read the parameter from the input stream.
// Note: Reads only the value, min_value, max_value, and prompt
void ParamBoolean::Read (istream & iss)
{
  // It is assumed that the name,mode,and type have already been read in.
  char temp[MAX_PARAM_LENGTH];
  int lngth;

  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if ((lngth = strlen (temp)) > 0)
    {
      if ((lngth > 2) && (temp[0] == '\"') && (temp[1] == ')') &&
	  (temp[lngth - 1] == '\"'))
	{
	  // indirection enclosed in quotes 
	  current = value = 0;
	  strcpy (val_redir, &temp[1]);
	  val_redir[lngth - 2] = '\0';
	  strcpy (cur_redir, val_redir);
	}
      else if (temp[0] == ')')
	{
	  // indirection 
	  current = value = 0;
	  strcpy (val_redir, temp);
	  strcpy (cur_redir, val_redir);
	}
      else if (strcmp (temp, "yes") == 0)
	{
	  current = value = true;
	}
    }
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if ((strlen (temp) > 0) && (strcmp (temp, "yes") == 0))
    min_value = true;
  iss.getline (temp, MAX_PARAM_LENGTH, ',');
  if ((strlen (temp) > 0) && (strcmp (temp, "yes") == 0))
    max_value = true;
  iss.getline (temp, MAX_PARAM_LENGTH, '\0');
  int temp_length;
  if ((temp_length = strlen (temp)) > 0)
    {
      if ((temp[0] == '"') && (temp[--temp_length] == '"'))
	{
	  temp[temp_length] = '\0';
	  prompt = &temp[1];
	}
    }
}

ostream & operator<< (ostream & oss, ParamItem &obj)
{
  obj.Print(oss);
  return oss;
}

istream & operator>> (istream & iss, ParamItem &obj)
{
  obj.Read(iss);
  return iss;
}
