# $Id$

package Data::ObjectDriver::Driver::Cache::Apache;
use strict;
use warnings;

use base qw( Data::ObjectDriver::Driver::BaseCache );

sub init {
    my $driver = shift;
    my %param  = @_;
    $param{cache} ||= 1; # hack
    $driver->SUPER::init(%param);
}

sub r {
    my $driver = shift;
    if ($INC{"mod_perl.pm"}) {
        return Apache->request;
    } elsif ($INC{"mod_perl2.pm"}) {
        return Apache2::RequestUtil->request;
    } else {
        die "Not running on mod_perl environment.";
    }
}

sub get_from_cache {
    my $driver = shift;
    my $r = $driver->r or return;

    $driver->start_query('APACHECACHE_GET ?', \@_);
    my $ret = $r->pnotes($_[0]);
    $driver->end_query(undef);

    return if !defined $ret;
    return $ret;
}

sub add_to_cache {
    my $driver = shift;
    my $r = $driver->r or return;

    $driver->start_query('APACHECACHE_ADD ?,?', \@_);
    my $ret = $r->pnotes($_[0], $_[1]);
    $driver->end_query(undef);

    return if !defined $ret;
    return $ret;
}

sub update_cache {
    my $driver = shift;
    my $r = $driver->r or return;

    $driver->start_query('APACHECACHE_SET ?,?', \@_);
    my $ret = $r->pnotes($_[0], $_[1]);
    $driver->end_query(undef);

    return if !defined $ret;
    return $ret;
}

sub remove_from_cache {
    my $driver = shift;
    my $r = $driver->r or return;

    $driver->start_query('APACHECACHE_DELETE ?', \@_);
    my $ret = delete $r->pnotes->{$_[0]};
    $driver->end_query(undef);

    return if !defined $ret;
    return $ret;
}

1;

__END__

=head1 NAME

Data::ObjectDriver::Driver::Cache::Apache - object driver for caching objects in Apache's request space

=head1 SYNOPSIS

    package MyObject;
    use base qw( Data::ObjectDriver::BaseObject );

    __PACKAGE__->install_properties({
        ...
        driver => Data::ObjectDriver::Driver::Cache::Apache->new(
            fallback => Data::ObjectDriver::Driver::Cache::Memcached->new(
                cache    => Cache::Memcached->new({ servers => \@MEMCACHED_SERVERS }),
                fallback => Data::ObjectDriver::Driver::DBI->new( @$DBI_INFO ),
            ),
        ),
        ...
    });

    1;

=head1 DESCRIPTION

I<Data::ObjectDriver::Driver::Cache::Apache> provides automatic caching of
retrieved objects in the per-request memory space of your Apache mod_perl
processes, when used in conjunction with your actual object driver. It can be
used to provide even faster results over memcached when requesting objects that
have already been requested during the same request by some other part of your
application, at the cost of the memory necessary to store the objects.

If your models can be used in an Apache mod_perl application as well as another
context such as a command line shell, consider replacing the Apache layer of
your caching with a C<Data::ObjectDriver::Driver::Cache::RAM> layer when Apache
is not available. See L<Data::ObjectDriver::Driver::Cache::Apache>.

=head1 USAGE

=over 4

=item * Data::ObjectDriver::Driver::Cache::Apache->new(%params)

Required members of C<%params> are:

=over 4

=item * C<fallback>

The C<Data::ObjectDriver> object driver from which to request objects that are
not found in the Apache process cache.

=back

=back

=head1 DIAGNOSTICS

The Apache driver provides integration with the C<Data::ObjectDriver> debug and
profiling systems. As these systems are designed around SQL queries, synthetic
queries are logged to represent caching operations. The operations generated by
this driver are:

=over 4

=item * C<APACHECACHE_GET ?>

Retrieve an object. The argument is the cache key for the requested object.

=item * C<APACHECACHE_ADD ?,?>

Add an object to the cache. The arguments are the cache key for the object and
the flattened representation of the object to cache.

=item * C<APACHECACHE_SET ?,?>

Put an object in the cache. The arguments are the cache key for the object and
the flattened representation of the object to cache.

=item * C<APACHECACHE_DELETE ?>

Remove an object from the cache. The argument is the cache key for the object
to invalidate.

=back

=head1 SEE ALSO

L<Apache>, L<Apache2::RequestUtil>

=head1 LICENSE

I<Data::ObjectDriver> is free software; you may redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHOR & COPYRIGHT

Except where otherwise noted, I<Data::ObjectDriver> is Copyright 2005-2006
Six Apart, cpan@sixapart.com. All rights reserved.

=cut
